% -------------------------------------------------------------------------
% -----------------------------MAIN ROUTINE--------------------------------
% -------------------------------------------------------------------------

% Algorithm reconstructs rho and nu from the 1D acoustic system 
%
%   1/(rho*nu^2)*dt_p = dx_v+source_1   in (0,1)x(0,T) 
%            rho*dt_v = dx_p+source_2   in (0,1)x(0,T)
%              p(0,t) = p(1,t) = 0      for t in (0,T)
%              p(x,0) = v(x,0) = 0      for x in (0,1)
%
% via REGINN^{\infty} given (noisy) wave fields p and v (provided delta>0). 
% Input detail: 
% - Use t*x instead of x*t to define input functions
% - Noise level is relative, i.e. ||(p,v)-(fdelta1,fdelta2)||_H = delta*||(p,v)||
% - If tau = 0, regularization aborts according to discretization limit (=N_x)
% 
%   With this code Figure 2 of the following manuscript was produced under Matlab(R2021a):
%   Pieronek, Rieder: On the iterative regularization of 
%                     non-linear illposed problems in $L^{\infty}$
%                     https://doi.org/10.5445/IR/1000140578
%

    format long;

% Input parameters
    % regularization specific
    mu=0.7; % initial mu
    mu_max=0.999;
    tau=1.1;
    gamma=0.8; 
    delta=0.0; % relative noise level 
    
    % discretization specific 
    n_min=power(2,2); % minimal parameter resolution size to start with (1<=n_min)
    N_t=100; % total number of time points for pdepe (>2), time step size = 1/(N_t-1) with midpoint covering time boundary
    N_x=300; % total number of spatial nodes for pdepe (n_min<N_x), cell size = 1/(N_x-1) with midpoint covering spatial boundary       
    eps=0.99999; % refine discretization if J_new/J_old exceeds eps after gradient step 
    
    % data and solution
    rho_jump=1.2;
    nu_jump=0.9;
    rho_0=@(x) 1*x.^0;% piecewise_eval(x,[1/3 2/3],{1,rho_jump,1});%    
    nu_0=@(x) 1*x.^0;% piecewise_eval(x,[1/3 2/3],{1,nu_jump,1});%
    p=@(t,x) 100*t*x.*(x-1); % exact p wave field
    v=@(t,x) 100*t*sin(pi/2*x); % exact v wave field
    source_1=@(t,x) 100*(piecewise_eval(x,[1/3-.1 2/3-.1],{1,1/(rho_jump),1}).*piecewise_eval(x,[1/3+.1 2/3+.1],{1,1/(nu_jump^2),1}).*x.*(x-1)-t*cos(pi/2*x)*pi/2); % computed from data and solution
    source_2=@(t,x) 100*(piecewise_eval(x,[1/3-.1 2/3-.1],{1,rho_jump,1}).*sin(pi/2*x)-t*(2*x-1));

    % other  
    T=1.0; % time interval
    paramin=0.1; % minimal pointwise parameter value = domain of definition 
    pY=2; % exponent of data space Y=L^pY(0,T,L^pY)
    rng(1); % seed=1 for noise generation
    C=1.1; % norm equivalence constant C_{\infty}
        
%%     

% Noise (Gaussian), relative to (p,v)    
    noise1 = randn(N_t,N_x);
    noise2 = randn(N_t,N_x);
    noise1= noise1/sqrt(Ynorm(noise1,N_x,N_t,T,pY)^2+Ynorm(noise2,N_x,N_t,T,pY)^2);
    noise2= noise2/sqrt(Ynorm(noise1,N_x,N_t,T,pY)^2+Ynorm(noise2,N_x,N_t,T,pY)^2);
    
% Plot references
    rho_fun =@(x) piecewise_eval(x,[1/3-.1 2/3-.1],{1,rho_jump,1});
    nu_fun =@(x) piecewise_eval(x,[1/3+.1 2/3+.1],{1,nu_jump,1});
    rho = rho_fun(linspace(0,1,N_x));
    nu = nu_fun(linspace(0,1,N_x));
    
% Meshes and discretized input functions
    xmesh = linspace(0,1,N_x);
    tspan = linspace(0,T,N_t); 
    source1=source_1(tspan.',xmesh);
    source2=source_2(tspan.',xmesh);
    normalized=sqrt(Ynorm(p(tspan.',xmesh),N_x,N_t,T,pY)^2+Ynorm(v(tspan.',xmesh),N_x,N_t,T,pY)^2);
    fdelta1=p(tspan.',xmesh)+delta*normalized*noise1;
    fdelta2=v(tspan.',xmesh)+delta*normalized*noise2;
    x0_1=rho_0(xmesh); 
    x0_2=nu_0(xmesh); 
    if min(x0_1)<=paramin || min(x0_2)<=paramin
        disp('Error: Initial guess must be larger than "paramin" pointwise!');
        return
    else
        figure(1)
        plot(xmesh,x0_1,'linewidth',2)
        hold on 
        plot(xmesh,rho,'linewidth',2)
        axis([0 1 0.9 1.3])
        yticks(linspace(0.9,1.3,5))
        ax=gca;
        ax.XAxis.FontSize = 12;
        ax.YAxis.FontSize = 12;
        title('Initial guess for \rho')
        hold off
        figure(2)
        plot(xmesh,x0_2,'linewidth',2)
        hold on 
        plot(xmesh,nu,'linewidth',2)
        axis([0 1 0.8 1.2])
        yticks(linspace(0.8,1.2,5))
        ax=gca;
        ax.XAxis.FontSize = 12;
        ax.YAxis.FontSize = 12;
        title('Initial guess for \nu')
        hold off
        drawnow
    end
    
% ALGORITHM START
    % Initialization
    m=0; % counter for Newton steps
    nk=0; % couunter for internal gradient steps
    xm_1=x0_1;
    xm_2=x0_2;
    penalty1=0;
    penalty2=0;
    n=n_min;
    p=ceil(pn(C,n)); 
    pdefun = @(x,t,u,dudx) pde_system(x,t,dudx,x0_1,x0_2,source1,source2,N_x,N_t,T);
    sol = pdepe(0,pdefun,@pdeic,@pdebc,xmesh,tspan);
    bm_delta1=fdelta1-sol(:,:,1);
    bm_delta2=fdelta2-sol(:,:,2);
    bm_norm=sqrt(Ynorm(bm_delta1,N_x,N_t,T,pY)^2+Ynorm(bm_delta2,N_x,N_t,T,pY)^2);
             
while bm_norm/normalized>(tau*delta) % normalization since delta is given as relative noise
    % dynamic mu
    if (m>1)
        if (nk>= nk_old)
            mu=min([mu_max 1-(nk_old/nk)*(1-mu)]);
        else
            mu=0.9*mu;
        end
    end
    if (m==0)
        disp(['Newton step ',num2str(m),' at discretization level ',num2str(n),':  nonlinear residuum = ',num2str(bm_norm,12),' (mu = ',num2str(mu),')']);
    else
        disp(['Newton step ',num2str(m),' at discretization level ',num2str(n),':  nonlinear residuum = ',num2str(bm_norm,12),' (mu = ',num2str(mu),', gradient steps = ',num2str(nk),')']);
    end
    nk_old=nk;  
    nk=0;
    s1=zeros(1,N_x); % Initialize inner loop after each Newton step with s1=s2=0
    s2=zeros(1,N_x); 
    dsol=zeros(N_t,N_x,2); 
    dt_sol1= dt(sol(:,:,1),N_x,N_t,T);
    dt_sol2= dt(sol(:,:,2),N_x,N_t,T); 
    alpha_m=bm_norm^2/gamma^2;    
    J_old=bm_norm^2+alpha_m*(penalty1^2+penalty2^2);
    
    % Newton step criterion
    while(J_old > mu^2*bm_norm^2)
    nk=nk+1;     
        % Compute gradient (residual + penalty part)
        % - penalty part: normalized duality mapping
            if penalty1==0
                Jpn1=zeros(1,N_x);
            else 
                Jpn1=penalty1*power(abs(s1+xm_1-x0_1)/penalty1,p-1).*sign(s1+xm_1-x0_1); % (notation avoids NaN=infty/infty for large p)
            end
            if penalty2==0
                Jpn2=zeros(1,N_x);
            else 
                Jpn2=penalty2*power(abs(s2+xm_2-x0_2)/penalty2,p-1).*sign(s2+xm_2-x0_2); 
            end
        % - residual part: solve adjoint equation 
            af1=flipud(dsol(:,:,1)-bm_delta1); % time inversion for adjoint source term
            af2=flipud(dsol(:,:,2)-bm_delta2); 
            a_norm=sqrt(Ynorm(af1,N_x,N_t,T,pY)^2+Ynorm(af2,N_x,N_t,T,pY)^2)^(2-pY);
            af1=abs(af1).^(pY-1).*sign(af1)/a_norm; % use duality mapping for non-Hilbert case
            af2=abs(af2).^(pY-1).*sign(af2)/a_norm;
            apdefun = @(x,t,u,dudx) pde_system(x,t,dudx,-xm_1,xm_2,af1,af2,N_x,N_t,T);
            asol = pdepe(0,apdefun,@pdeic,@pdebc,xmesh,tspan); 
            asol(:,:,1)=flipud(asol(:,:,1)); % time re-inversion of adjoint solution
            asol(:,:,2)=flipud(asol(:,:,2));
            dtpw=(dot(dt_sol1(2:(N_t-1),:),asol(2:(N_t-1),:,1))...
                  +0.5*dt_sol1(1,:).*asol(1,:,1)+0.5*dt_sol1(N_t,:).*asol(N_t,:,1))*T/(N_t-1); % adjoint time integral   
            dtvw=(dot(dt_sol2(2:(N_t-1),:),asol(2:(N_t-1),:,2))...
                  +0.5*dt_sol2(1,:).*asol(1,:,2)+0.5*dt_sol2(N_t,:).*asol(N_t,:,2))*T/(N_t-1);
            adj1=dtvw-dtpw./(xm_1.^2.*xm_2.^2);  
            adj2=-2*dtpw./(xm_1.*xm_2.^3); 
            grad_1=adj1+alpha_m*Jpn1;
            grad_2=adj2+alpha_m*Jpn2;
            update1=discretegrad(n,N_x,grad_1); % averaging as adjoint of embedding operator 
            update2=discretegrad(n,N_x,grad_2);
            update1_fun=@(x) piecewise_eval(x,linspace(1/n,1-1/n,n-1),num2cell(update1)); 
            update2_fun=@(x) piecewise_eval(x,linspace(1/n,1-1/n,n-1),num2cell(update2)); 
            update1_mesh=update1_fun(xmesh); % = gradient in fine mesh coordinates
            update2_mesh=update2_fun(xmesh);
            fupdate1=dt_sol1.*(ones(N_t,1)*(update1_mesh./(xm_1.^2.*xm_2.^2)+2*update2_mesh./(xm_1.*xm_2.^3))); 
            fupdate2=dt_sol2.*(-ones(N_t,1)*update1_mesh);
            updefun = @(x,t,u,dudx) pde_system(x,t,dudx,xm_1,xm_2,fupdate1,fupdate2,N_x,N_t,T);
            updatesol = pdepe(0,updefun,@pdeic,@pdebc,xmesh,tspan);  
        
        % Determine k in (-k*grad+s) for descent, i.e. J_new < J_old
        s1_old=s1;
        s2_old=s2;
        dsol1_old=dsol(:,:,1);
        dsol2_old=dsol(:,:,2);
        J_new=J_old;       
        k=1; 
        while(J_new >= J_old)
            s1=-k*update1_mesh+s1_old;                            
            s2=-k*update2_mesh+s2_old;
            if min(s1+xm_1)>0 && min(s2+xm_2)>0 % gradient descent with armijo linesearch
                dsol(:,:,1)=-k*updatesol(:,:,1)+dsol1_old;
                dsol(:,:,2)=-k*updatesol(:,:,2)+dsol2_old;
                linres=Ynorm(dsol(:,:,1)-bm_delta1,N_x,N_t,T,pY)^2+Ynorm(dsol(:,:,2)-bm_delta2,N_x,N_t,T,pY)^2;
                penalty1=pnorm(s1+xm_1-x0_1,N_x,p);
                penalty2=pnorm(s2+xm_2-x0_2,N_x,p);
                J_new=linres+alpha_m*(penalty1^2+penalty2^2);
            else
                J_new=J_old;
            end
            k=k/2;
            if k<realmin
                s1=s1_old;                            
                s2=s2_old;
                dsol(:,:,1)=dsol1_old;
                dsol(:,:,2)=dsol2_old;
                J_new=J_old;
                break;
            end
        end
        
        % Decide if gradient step was good enough (wrt eps), else discretization needs to be refined
        if (J_new/J_old > eps)
            if(n<=(N_x-1)/2)
                n=2*n;
                p=ceil(pn(C,n));
                penalty1=pnorm(s1+xm_1-x0_1,N_x,p);
                penalty2=pnorm(s2+xm_2-x0_2,N_x,p);
                J_old = linres+alpha_m*(penalty1^2+penalty2^2); 
            else
                xm_1=xm_1+s1;                                              
                xm_2=xm_2+s2;
                pdefun = @(x,t,u,dudx) pde_system(x,t,dudx,xm_1,xm_2,source1,source2,N_x,N_t,T);
                sol = pdepe(0,pdefun,@pdeic,@pdebc,xmesh,tspan);      
                bm_delta1=fdelta1-sol(:,:,1);
                bm_delta2=fdelta2-sol(:,:,2);
                bm_norm=sqrt(Ynorm(bm_delta1,N_x,N_t,T,pY)^2+Ynorm(bm_delta2,N_x,N_t,T,pY)^2);
                disp(['Pseudo-Newton step ',num2str(m+1),' at discretization level ',num2str(n),':  nonlinear residuum = ',num2str(bm_norm,12),' (',num2str(nk),' gradient steps)']);
                disp('Termination: discretization limit reached');
                if bm_norm/normalized<=(tau*delta)
                   disp('(Outer condition fulfilled via Pseudo Newton step)'); 
                end    
                figure(1)
                plot(xmesh,xm_1,'linewidth',2)
                hold on 
                plot(xmesh,rho,'linewidth',2)
                axis([0 1 0.9 1.3])
                yticks(linspace(0.9,1.3,5))
                ax=gca;
                ax.XAxis.FontSize = 12;
                ax.YAxis.FontSize = 12;
                title('Pseudo-updated \rho')
                hold off   
                figure(2)
                plot(xmesh,xm_2,'linewidth',2)
                hold on 
                plot(xmesh,nu,'linewidth',2)
                axis([0 1 0.8 1.2])
                yticks(linspace(0.8,1.2,5))
                ax=gca;
                ax.XAxis.FontSize = 12;
                ax.YAxis.FontSize = 12;
                title('Pseudo-updated \nu')
                hold off
                drawnow
                return
            end
        else 
            J_old = J_new;
        end
    end     
    
    % Newton update
    m=m+1;
    xm_1=xm_1+s1;                                              
    xm_2=xm_2+s2;
    pdefun = @(x,t,u,dudx) pde_system(x,t,dudx,xm_1,xm_2,source1,source2,N_x,N_t,T);
    sol = pdepe(0,pdefun,@pdeic,@pdebc,xmesh,tspan);      
    bm_delta1=fdelta1-sol(:,:,1);
    bm_delta2=fdelta2-sol(:,:,2);
    bm_norm=sqrt(Ynorm(bm_delta1,N_x,N_t,T,pY)^2+Ynorm(bm_delta2,N_x,N_t,T,pY)^2);
    
    % Plot Newton update
    figure(1)
    plot(xmesh,xm_1,'linewidth',2)
    hold on 
    plot(xmesh,rho,'linewidth',2)
    axis([0 1 0.9 1.3])
    yticks(linspace(0.9,1.3,5))
    ax=gca;
    ax.XAxis.FontSize = 12;
    ax.YAxis.FontSize = 12;
    title('Updated \rho')
    hold off   
    figure(2)
    plot(xmesh,xm_2,'linewidth',2)
    hold on 
    plot(xmesh,nu,'linewidth',2)
    axis([0 1 0.8 1.2])
    yticks(linspace(0.8,1.2,5))
    ax=gca;
    ax.XAxis.FontSize = 12;
    ax.YAxis.FontSize = 12;
    title('Updated \nu')
    hold off
    drawnow
end
disp(['Newton step ',num2str(m),' at discretization level ',num2str(n),':  nonlinear residuum = ',num2str(bm_norm,12),' (mu = ',num2str(mu),', ',num2str(nk),' gradient steps)']);
disp('Termination: outer condition fulfilled');
% ALGORITHM END


%%   

% -------------------------------------------------------------------------
% -------------------FUNCTION DEFINITIONS----------------------------------
% ------------------------------------------------------------------------- 

function ixt = xt2mesh(xt,Nxt,bnd)
% Convert spatial or time coordinate xt within (0,bnd) to mesh index ixt
% ..0..](....]   ....   (....](..1..
ixt=ceil((xt+0.5*bnd/(Nxt-1))*(Nxt-1)/bnd);
end
% -------------------------------------------------------------------------
function val = discretegrad(nn,Nx,grad)
% Average gradient from fine to parameter mesh, assumes Nx > nn

xup=linspace(1/nn,1,nn);
xun=linspace(0,1-1/nn,nn);
ixtup=xt2mesh(xup,Nx,1);
ixtun=xt2mesh(xun,Nx,1);
vec=zeros(1,nn);
for ii=1:nn
    if (ixtup(ii)-ixtun(ii))>1
        inner=sum(grad(1,(ixtun(ii)+1):(ixtup(ii)-1)));
    else
        inner=0;
    end
    rbnd=(ixtun(ii)-0.5)/(Nx-1);%          xun=# ]=rbnd      lbnd=( #=xup         
    un=(rbnd-xun(ii))*(Nx-1);   % 0.]  ....  (.#.](...] .... (...](.#.]  ....  (.1  
    lbnd=(ixtup(ii)-1.5)/(Nx-1);  
    up=(xup(ii)-lbnd)*(Nx-1);  
    vec(ii)=(inner+un*grad(ixtun(ii))+up*grad(ixtup(ii)))*nn/(Nx-1);
end   
val=vec;
end
% -------------------------------------------------------------------------
function [c,f,s] = pde_system(xx,tt,dudx,pmeter1,pmeter2,ff1,ff2,Nx,Nt,TT) 
% Specify acoustic PDE system in 1D with u =(p,v): 1/(rho*nu^2)*dt_p=dx_v+ff1 
%                                                  rho*dt_v=dx_p+ff2
it = xt2mesh(tt,Nt,TT);
ix = xt2mesh(xx,Nx,1);
c = [1./(pmeter1(ix).*pmeter2(ix).^2); pmeter1(ix)];
f = [0; 0];
s = [ff1(it,ix)+dudx(2); ff2(it,ix)+dudx(1)];
end
% -------------------------------------------------------------------------
function u0 = pdeic(~) 
% Set zero initial conditions
u0 = [0; 0];
end
% -------------------------------------------------------------------------
function [pl,ql,pr,qr] = pdebc(~,ul,~,ur,~) 
% Set Dirichlet boundary conditions for p, none for v
pl = [ul(1); 0];
ql = [0; 1];
pr = [ur(1); 0]; 
qr = [0; 1];
end
% -------------------------------------------------------------------------
function val = pn(Cn,nn)
% Find L^{pp} equivalence constant Cn wrt L^\infty at discretization level nn
    val=log(nn)/log(Cn);
end
% -------------------------------------------------------------------------
function val = pnorm(fun,Nx,pp)
% Compute L^{pp}-norm over default domain (0,1) 
    val=nthroot((norm(fun(1,2:(Nx-1)),pp)^pp...
       +0.5*abs(fun(1,1))^pp+0.5*abs(fun(1,Nx))^pp)/(Nx-1),pp); 
    %val=norm(fun,pp)/nthroot((Nx-1),pp); % better conditioned for large pp?
end
% -------------------------------------------------------------------------
function val = Ynorm(fun,Nx,Nt,TT,pp)
% Compute qq-norm over space-time cylinder [0,1]x[0,TT] for array fun
    interior= norm(reshape(fun(2:(Nt-1),2:(Nx-1)),[],1),pp)^pp;
    tboundary=0.5*norm(fun(1,2:(Nx-1)),pp)^pp+0.5*norm(fun(Nt,2:(Nx-1)),pp)^pp;
    xboundary=0.5*norm(fun(2:(Nt-1),1),pp)^pp+0.5*norm(fun(2:(Nt-1),Nx),pp)^pp;
    corners=0.25*(abs(fun(1,1))^pp+abs(fun(Nt,1))^pp+abs(fun(Nt,Nx))^pp+abs(fun(1,Nx))^pp);
    val=nthroot((interior+tboundary+xboundary+corners)*TT/((Nx-1)*(Nt-1)),pp);
end
% -------------------------------------------------------------------------
function val = dt(fun,Nx,Nt,TT)
% Compute time derivative of space-time array fun
    tmp=zeros(Nt,Nx);
    tmp(2:(Nt-1),:)=(fun(3:Nt,:)-fun(1:(Nt-2),:))*(Nt-1)/(2*TT); % central differences in interior
    tmp(1,:)=(fun(2,:)-fun(1,:))*(Nt-1)/TT; % one-sided differences for boundary
    tmp(Nt,:)=(fun(Nt,:)-fun((Nt-1),:))*(Nt-1)/TT;
    val=tmp;
end
% -------------------------------------------------------------------------
function z = piecewise_eval(x,breakpoints,funs)
% PIECEWISE_EVAL: evaluates a piecewise function of x
% usage: y = PIECEWISE_EVAL(x,breakpoints,funs)
%
% arguments (input)
%  x    - vector or array of points to evaluate though the function
%  
%  breakpoints - list of n breakpoints, -inf and +inf are implicitly
%         the first and last breakpoints. A function with only two
%         pieces has only one explicit breakpoint. In the event that
%         you want to define a function with breakpoints [a,b,c],
%         and only two functions, but you do not care what happens
%         for x < a or x > b, then you should specify only the
%         breakpoint b. Alternatively, one could specify all 3
%         breaks, and force the function to return NaN above and
%         below those limits.
%
%         x(i) will be identified as falling in interval (j) if
%         break(j) <= x(i) < break(j+1)
%  
%  funs - cell array containing n+1 functions as scalar constants,
%         strings, anonymous functions, inline functions, or any
%         simple matlab function name.
%
%         Note: use .*, ./, .^ where appropriate in the function
%
%         These functions need not be differentiable or even
%         continuous across the breaks.
%
% arguments (output)
%  z    - evaluated function, result is same shape as x
%
% Example usage:
%  For       x < -5, y = 2
%  For -5 <= x < 0,  y = sin(x)
%  For  0 <= x < 2,  y = x.^2
%  For  2 <= x < 3,  y = 6
%  For  3 <= x,      y = inf
%
%  y = piecewise_eval(-10:10,[-5 0 2 3],{2,'sin(x)','x.^2',6,inf})
n=length(breakpoints);
% there must be n+1 funs for n breaks
if length(funs)~=(n+1)
  error 'funs and breakpoints are incompatible in size'
end
if any(diff(breakpoints)<=0)
  error 'Breakpoints must be both distinct and increasing'
end
% ensure the functions are feval-able
for i=1:(n+1)
  if ischar(funs{i})
    % A string. Make it a function
    f=inline(funs{i});
    funs{i} = f;
  elseif isa(funs{i},'function_handle') || isa(funs{i},'inline')
    % a function handle or an inline. do nothing.
  elseif isnumeric(funs{i}) | isnan(funs{i}) | isinf(funs{i})
    % A scalar value was supplied, may be NaN or inf.
    % Make it a function.
    funs{i}=@(x) funs{i};
  else
    % It must be something that feval can handle
    % directly, so leave funs{i} alone.
  end
end
% initialize as nans
z=nan(size(x));
% below the first break
k=(x<breakpoints(1));
z(k)=feval(funs{1},x(k));
left = k;
for i=2:n
  k=(~left) & (x<breakpoints(i));
  if any(k)
    z(k)=feval(funs{i},x(k));
    left = k | left;
  end
end
% over the top
k=(x>=breakpoints(end));
z(k)=feval(funs{end},x(k));
end

